<?php
namespace LiquidElementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Scheme_Color;
use Elementor\Scheme_Typography;
use Elementor\Utils;
use Elementor\Control_Media;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Background;
use Elementor\Repeater;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor heading widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 2.1.3
 */
class LD_Breadcrumb extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 2.1.3
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'lqd-breadcrumb';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve heading widget title.
	 *
	 * @since 2.1.3
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Liquid Breadcrumb', 'logistics-hub-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve heading widget icon.
	 *
	 * @since 2.1.3
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-product-breadcrumbs lqd-element';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.1.3
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'hub-core' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.3
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'breadcrumb', 'seo' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 2.1.3
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'general_section',
			[
				'label' => __( 'General', 'logistics-hub-core' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'breadcrumb_separator',
			[
				'label' => esc_html__( 'Separator', 'logistics-hub-core' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( '/', 'logistics-hub-core' ),
				'placeholder' => esc_html__( '/', 'logistics-hub-core' ),
				'selectors' => [
					'{{WRAPPER}} li:not(:last-child):after' => 'content: "{{VALUE}}"; margin-inline-start: 10px;',
				]
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'breadcrumb_typography',
				'selector' => '{{WRAPPER}}',
			]
		);

		$this->add_responsive_control(
			'justify_content',
			[
				'label' => esc_html__( 'Alignment', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'default' => '',
				'options' => [
					'' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-flex eicon-justify-start-h',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-flex eicon-justify-center-h',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-flex eicon-justify-end-h',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .breadcrumb' => 'text-align: {{VALUE}}; justify-content: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'breadcrumb_gap',
			[
				'label' => esc_html__( 'Gap', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem', 'custom' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .breadcrumb' => 'gap: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .breadcrumb > li:not(:last-child):after' => 'margin-inline-start: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'breadcrumb_color',
			[
				'label' => __( 'Active item color', 'logistics-hub-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}}' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'breadcrumb_link_color',
			[
				'label' => __( 'Link item color', 'logistics-hub-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'breadcrumb_link_hover_color',
			[
				'label' => __( 'Link item hover color', 'logistics-hub-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a:hover' => 'color: {{VALUE}};',
				],
			]
		);

        $this->end_controls_section();

	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 2.1.3
	 * @access protected
	 */
	protected function render() {

		$breadcrumb_args = array(
			'classes' => 'list-none p-0 m-0 flex flex-wrap items-center gap-10',
		);

		if ( class_exists('WooCommerce') ){
			$breadcrumb_args = array(
				'wrap_before' => '<div class="lqd-shop-topbar-breadcrumb"><nav class="mb-0 woocommerce-breadcrumb"><ul class="flex flex-wrap items-center gap-10 p-0 m-0 list-none breadcrumb">',
				'wrap_after'  => '</ul></nav></div>'
			);
		}

		?><div class="lqd-breadcrumb-wrapper"><?php

		if ( class_exists('WooCommerce') ){
			woocommerce_breadcrumb( $breadcrumb_args );
		} else {
			liquid_breadcrumb( $breadcrumb_args );
		}

		?></div><?php

	}

}
\Elementor\Plugin::instance()->widgets_manager->register( new LD_Breadcrumb() );